/*
 * Copyright (c) 2024, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */

package jdk.internal.event;

/**
 * A JFR event for forced updates written to files.  This event is mirrored in
 * {@code jdk.jfr.events.FileForceEvent } where the event annotations are
 * provided.  Some of the methods are replaced by generated
 * methods when jfr is enabled.  Note that the order of the arguments of the
 * {@link #commit(long, long, String, boolean)} method
 * must be the same as the order of the fields.
 */
public class FileForceEvent extends Event {

    // THE ORDER OF THE FOLLOWING FIELDS IS IMPORTANT!
    // The order must match the argument order of the generated commit method.
    public String path;
    public boolean metaData;

    /**
     * Helper method to offer the data needed to potentially commit an event.
     * The duration of the operation is computed using the current
     * timestamp and the given start time.  If the duration is meets
     * or exceeds the configured value (determined by calling the generated method
     * {@link #shouldCommit(long)}), an event will be emitted by calling
     * {@link #commit(long, long, String, boolean)}.
     *
     * @param start     timestamp of the start of the operation
     * @param path      the full pathname of the file
     * @param metaData  true if the file metadata is updated
     */
    public static void offer(long start, String path, boolean metaData) {
        long duration = timestamp() - start;
        if (shouldCommit(duration)) {
            commit(start, duration, path, metaData);
        }
    }

    /**
     * Actually commit an event.  The implementation is generated automatically.
     * The order of the fields must be the same as the parameters in this method.
     *
     * @param start     timestamp of the start of the operation
     * @param duration  time in nanoseconds to complete the operation
     * @param path      the full pathname of the file
     * @param metaData  true if the file metadata is updated
     */
    public static void commit(long start, long duration, String path, boolean metaData) {
        // Generated by JFR
    }

    /**
     * Determine if an event should be emitted.  The duration of the operation
     * must exceed some threshold in order to commit the event.  The implementation
     * of this method is generated automatically if jfr is enabled.
     *
     * @param duration  time in nanoseconds to complete the operation
     * @return  true if the event should be commited
     */
    public static boolean shouldCommit(long duration) {
        // Generated by JFR
        return false;
    }

    /**
     * Determine if this kind of event is enabled.  The implementation
     * of this method is generated automatically if jfr is enabled.
     *
     * @return true if this type of event is enabled, false otherwise
     */
    public static boolean enabled() {
        // Generated by JFR
        return false;
    }

    /**
     * Fetch the current timestamp in nanoseconds.  This method is used
     * to determine the start and end of an operation.  The implementation
     * of this method is generated automatically if jfr is enabled.
     *
     * @return  the current timestamp value
     */
    public static long timestamp() {
        // Generated by JFR
        return 0L;
    }
}
